  ' -----------------------------------------------------------
  ' Program for Micromite to communicate with the nRF24L01+
  ' 2.4GHz data transceiver module, via the SPI interface,
  ' for checking its capabilities.
  ' Written by Jim Rowe for Silicon Chip.
  ' Last revision 25/8/2017 at 9:20 am
  '
  ' NOTE that this program uses an additional Micromite SPI port
  ' to communicate with the nRF24L01+ module, using these pins:
  ' SPI Function          Micromite pin
  ' MOSI (SPI out)        Pin 21
  ' MISO (SPI in)         Pin 22
  ' SCK (SPI clock)       Pin 24
  ' CSN (chip select-bar) Pin 18
  ' CE (chip enable)      Pin 17
  ' The embedded C function code needed to support this additional
  ' SPI port can be found at the end of the program.
  '
  ' NOTE also that the SPI interface of the nRF24L01+ has the
  ' following requirements:
  ' 1. The CSN line must be LOW for serial commands & write data
  '    on the MOSI line to be clocked into the nRF24L01+, and also
  '    for device STATUS and read data to be clocked out of the
  '    MISO line. This must be controlled by the program.
  ' 2. The CE line should be taken HIGH in RX mode or pulsed high
  '    for at least 10us in TX mode (after sending TX data to chip).
  '    If CE is held LOW, the nRF24L01+ stays in standby mode.
  '    (Basically, the CE line is used to start the radio transceiver
  '    section of the nRF24L01+ chip, and/or to hold it ON.)
  ' 3. the contents of the nRF24L01+ STATUS register (& read data,
  '    if any) are shifted out of the MISO pin simultaneously with
  '    the SPI command word & write data shifting in via the MOSI pin.
  '    But note that this is the existing STATUS register contents,
  '    NOT the status after the command/data is written.
  ' 4. Commands, status bytes and data are clocked into and out of
  '    the nRF24L01 on the RISING edges of the SCK pulses.
  ' 5. The SPI commands and data sent to the nRF24L01+ must be in
  '    the following formats:
  '    Command words: MSbit first, LSbit last
  '    Data words (multi bytes): LSB first & MSB last, with each
  '    byte sent MSbit first.
  ' 6. For correct operation, the SPI MODE should be set for 00
  '    -- i.e., CPOL = 0 and CPHA = 0.
  ' 7. We are using MMBasic's Speed 0 for the SPI transactions, which
  '    corresponds to a bit rate/SCK frequency of 1MHz when the Micromite
  '    CPU clock is 40MHz.
  ' NOTE also that nRF24L01 registers 08h(OBSERVE_TX), 09h(RPD),
  '    0Ah(RX_ADDR_P0)-0Fh(RX_ADDR_P5), 10h(TX_ADDR), 13h(RX_PW_P2)-16h(RX_PW_P5)
  '    and 17h(FIFO_STATUS) are not written to during initialisation, but
  '    are left with their default reset contents.
  ' NOTE further that for compatibility with modules connected to an Arduino Uno
  '   programmed using the RF24 library, the frequency channel is set to 4Ch (76).
  '   This is also the reason for using the air data rate of 1Mb/s.
  ' ---------------------------------------------------------------------------  
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  DIM AS INTEGER index               ' counter
  DIM AS INTEGER outData, inData     ' variables
  DIM AS INTEGER StatusByte = 0      ' STATUS register contents returned from SPI
  DIM AS INTEGER IntFlag = 0         ' flag to show if screen has been touched
  DIM AS INTEGER SettChg = 0         ' flag to show when settings have changed
  DIM AS INTEGER RadioNum = 0        ' radio number (0 or 1), initially 0
  DIM AS INTEGER RxTxMode = 0        ' mode flag (0 = Rx mode, 1 = Tx mode)
  DIM AS INTEGER RawTVal             ' variable for time value (raw) from TIMER
  DIM AS INTEGER PrepTime = 0        ' timer ms in 24 bit reversed byte order
  DIM AS INTEGER PTmask              ' mask used in reversing byte order
  DIM AS INTEGER MaxRTxbit = 0       ' flag for maximum no of Tx retransmits
  DIM AS INTEGER TxFIFOFul = 0       ' flag to show if Tx FIFO is full or not
  DIM AS INTEGER RxPipeNum = 0       ' data pipe number for available Rx data
  DIM AS INTEGER RxDataRdy = 0       ' Rx data ready flag bit (1 = data ready)
  DIM AS INTEGER TxDatSent = 0       ' Tx data sent flag bit(1 = data sent)
  DIM AS INTEGER RxDataWrd = 0       ' Rx payload data word (24 bits)
  DIM AS INTEGER RRxDataWd = 0       ' Rx payload data word with reversed byte order

  Dim AS INTEGER RegData(12) ' array for nRF24L01 control register initialise cmds
  RegData(0) = &H2103   ' write>EN_AA: enable Auto Ack function for P0 & P1 only
  RegData(1) = &H2203   ' write>EN_RXADDR: enable Rx Addr's for P0 & P1
  RegData(2) = &H2303   ' write>SETUP_AW: set Rx/Tx addr width to 5 bytes, all pipes
  RegData(3) = &H2433   ' write>SETUP_RETR: set ARD = 1000us, allow 3 retransmits
  RegData(4) = &H254C   ' write>RF_CH: set to freq channel 4Ch/76d (= 2.476GHz)
  RegData(5) = &H2604   ' write>RF_SETUP: set to 1Mb/s data rate & -6dBm Tx pwr
  RegData(6) = &H3103   ' write>RX_PW_P0: allow 3 bytes Rx payload (P0)
  RegData(7) = &H3203   ' write>RX_PW_P1: allow 3 bytes Rx payload (P1)
  RegData(8) = &H3C03   ' write>DYNPD: enable dyn payload length for P0 & P1
  RegData(9) = &H3D06   ' write>FEATURE: enable EN_ACK_PAY, EN_DPL
  RegData(10) = &HE100  ' command to flush Tx FIFO (in TX mode)
  RegData(11) = &HE200  ' command to flush Rx FIFO (in RX mode)
  ' ---------------------------------------------------------------------------
  
  CONST LSBMask24 = &H0000FF  ' mask for LS byte of a 24-bit word
  CONST NSBMask24 = &H00FF00  ' mask for LS+1 byte of a 24-bit word
  CONST TSBMask24 = &HFF0000  ' mask for LS+2 byte of a 24-bit word
  CONST WTxPLHedr = &HA0000000  ' W_TX_PAYLOAD cmd header (cmd byte A0)
  CONST RRxPLHedr = &H61000000  ' R_RX_PAYLOAD header (cmd byte 61)
  CONST WAckPLhdr = &HA8      ' header for RX ack payload

  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  
  SETPIN 22, DIN          ' declare pin 22 an input (for MISO)
  PIN(21) = 0             ' then initialise pin 21 to low and
  SETPIN 21, DOUT         ' declare it a digital output (MOSI)
  PIN(24) = 0             ' next set pin 24 to low and
  SETPIN 24, DOUT         ' declare it a digital output (SCK)
  PIN(17) = 0             ' then set pin 17 to low and
  SETPIN 17, DOUT         ' declare it a digital output (CE)
  PIN(18) = 1             ' then set pin 18 to high and
  SETPIN 18, DOUT         ' declare it a digital output (CSN)
  SETPIN 15, INTL, TchInt ' call TchInt to set flag when screen touched
  TIMER = 0               ' reset timer so it starts
  
  ' first write to the nRF24L01's STATUS register, to reset it
  outData = &H270E
  Send16
  ' then write to the CONFIG register, to power it up in PRX mode
  outData = &H207F    ' (also mask ints, enable CRC and set CRC for 16 bits)
  Send16              ' send it
  Pause 5             ' now delay 5ms so chip's XO can settle after PWR_UP
  ' next initialise the other nRF24L01 registers
  PRINT "Initialising nRF24L01 registers. "
  For index = 0 To 11
    outData = RegData(index)  ' make outData the next 16-bit word to send
    Send16                    ' then send it
  Next
  ' next initialise the RX_ADDR_P0 and TX_ADDR registers
        outData = &H2AE7E7E7E7E7    ' set RX_ADDR_P0
        Send48                      ' to E7E7E7E7E7
        outData = &H30E7E7E7E7E7    ' also set TX_ADDR to
        Send48                      ' the same value  
  PRINT "Done!"               ' confirm it was done
  PRINT "Now in RECEIVE mode (Radio 0)"   ' now advise user in Rx mode
  ClrStatReg          ' reset Status register
  '
  ' -----------------------------------------------------------------
  ' main program loop starts here
  ShowOpngScrn          ' first show the opening screen
  DO
    IF IntFlag = 1 THEN CheckBtn    ' if screen was touched, go check it out
    
    IF SettChg = 1 THEN             ' then if settings were changed, update
      IF RxTxMode = 1 THEN
        outData = &H207E            ' Tx mode, so clear PRX bit in CONFIG
        Send16                      ' register (CONFIG bit 0)
      ELSE
        outData = &H207F            ' Rx mode, so set PRX bit in CONFIG
        Send16                      ' register (CONFIG bit 0)
      END IF
      SettChg = 0                   ' finally clear settings changed flag
    END IF
    ' jumps here if no settings have been changed (SettChg = 0)
    IF RxTxMode = 1 THEN            ' if we are in PTX mode
      PIN(17) = 0                   ' make sure CE pin is low
      ClrStatReg                    ' then clear STATUS flag bits and
      FlushTxFIFO                   ' also flush the Tx FIFO
      GoSendTime                    ' then go send a time value
    ELSE                            ' but if we are in PRX mode
      PIN(17) = 1                   ' make sure CE pin is high
      GoListen                      ' then go listen for new Rx data
    END IF
  PAUSE 1000                      ' set to loop about once per second
  LOOP
END ' end of main part of program; subroutines and functions follow
  
  ' *****************************************************************
  ' subroutine to show the opening LCD screen with buttons
SUB ShowOpngScrn 
  IntFlag = 0     ' clear interrupt flag to begin
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), DBlue
  TEXT MM.HRes/2, MM.VRes/8, "SILICON CHIP", CM, 1, 3, Red, DBlue
  TEXT MM.HRes/2, MM.VRes/4, "nRF24L01 Module Checkout Program", CM, 1, 1, White, DBlue
  TEXT MM.HRes/2, MM.VRes*3/8, "Configure Radio:", CM, 1, 2, Green, DBlue
  RBOX 4, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Red
  TEXT MM.HRes/4, MM.VRes*5/8, "RADIO #0", CM, 1, 2, White, Red  
  RBOX 4, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes/4, MM.VRes*7/8, "RADIO #1", CM, 1, 2, Black, Bone  
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Red
  TEXT MM.HRes*3/4, MM.VRes*5/8, "RECEIVE", CM, 1,2, White, Red  
  RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*7/8, "TRANSMIT", CM, 1,2, Black, Bone
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to set IntFlag when touchscreen is touched
SUB TchInt
  IntFlag = 1
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to check nRF24L01+ status register & update flags
SUB FetchnCheckStatus
  outData = &HFF            ' prepare NOP cmd (to read status)
  Send8                     ' send it, then 
  StatusByte = inData       ' get StatusByte from inData
  RxDataRdy = (StatusByte AND &H40) >> 6  ' Rx data ready flag (1 = new data)
  TxDatSent = (StatusByte AND &H20) >> 5  ' Tx data sent flag (1 = data sent)
  MaxRTxbit = (StatusByte AND &H10) >> 4  ' Max number of Tx retransmits
  RxPipeNum = (StatusByte AND &H0E) >> 1  ' pipe No (0-5) for Rx data available
  TxFIFOFul = (StatusByte AND &H01)       ' 1 = TX FIFO full, 0 = not full
END SUB   ' return with various flags updated
  ' -----------------------------------------------------------------
  ' subroutine to write an 8-bit word to one of the nRF24L01 registers
SUB Send8
  PIN(18) = 0              ' first bring CSN low to start transaction
  inData = SPIPort(22,21,24,outData,0,0,8) ' then send the 8-bit data
  PIN(18) = 1              ' then bring CSN high to end transaction
END SUB                    ' and return
  ' -----------------------------------------------------------------
  ' subroutine to write a 16-bit word to one of the nRF24L01 registers
SUB Send16
  PIN(18) = 0              ' first bring CSN low to start transaction
  inData = SPIPort(22,21,24,outData,0,0,16) ' then send the 16-bit data
  PIN(18) = 1              ' then bring CSN high to end transaction
END SUB                    ' and return
  ' -----------------------------------------------------------------
  ' subroutine to send a 32-bit word to the nRF24L01
SUB Send32
  PIN(18) = 0              ' first bring CSN low to start transaction
  inData = SPIPort(22,21,24,outData,0,0,32)  ' then send the 32-bit data
  PIN(18) = 1              ' then bring CSN high to end transaction
END SUB                    ' and return
  ' -----------------------------------------------------------------
  ' subroutine to send a 48-bit word to the nRF24L01
SUB Send48
  PIN(18) = 0              ' first bring CSN low to start transaction
  inData = SPIPort(22,21,24,outData,0,0,48)  ' then send the 48-bit data
  PIN(18) = 1              ' then bring CSN high to end transaction
END SUB                    ' and return
  ' -----------------------------------------------------------------
  ' subroutine called when screen is touched, to check whether it has
  ' been touched on either the Radio #0, Radio #1, RECEIVE or TRANSMIT
  ' buttons on the LCD (or not, in which case it just returns)
SUB CheckBtn
  IntFlag = 0       ' first clear interrupt flag
  IF TOUCH(Y) < MM.VRes/2 THEN
    SettChg = 0     ' no valid button pressed, so just return
    EXIT SUB        ' with SettChg still 0
  END IF
  IF TOUCH(X) < MM.HRes/2 THEN    ' but if touch is on lower LH side
    IF TOUCH(Y) < MM.VRes*3/4 THEN  ' and on upper button
      RadioNum = 0
      RBOX 4, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Red
      TEXT MM.HRes/4, MM.VRes*5/8, "RADIO #0", CM, 1, 2, White, Red  
      RBOX 4, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
      TEXT MM.HRes/4, MM.VRes*7/8, "RADIO #1", CM, 1, 2, Black, Bone  
      PRINT "This is now Radio 0"
    ELSE      ' must be on lower button
      RadioNum = 1
      RBOX 4, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone      
      TEXT MM.HRes/4, MM.VRes*5/8, "RADIO #0", CM, 1, 2, Black, Bone
      RBOX 4, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Red      
      TEXT MM.HRes/4, MM.VRes*7/8, "RADIO #1", CM, 1, 2, White, Red
      PRINT "This is now Radio 1"
    END IF
  ELSE    ' touch must be on lower RH side
    IF TOUCH(Y) < MM.VRes*3/4 THEN  ' if it's on upper button
      RxTxMode = 0                  ' switch to Rx mode    
      RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Red
      TEXT MM.HRes*3/4, MM.VRes*5/8, "RECEIVE", CM, 1,2, White, Red  
      RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Bone
      TEXT MM.HRes*3/4, MM.VRes*7/8, "TRANSMIT", CM, 1,2, Black, Bone
      PRINT "Now in RECEIVE mode..."   ' advise user now in Rx mode
    ELSE                  ' must be on lower button, so
      RxTxMode = 1        ' switch to Tx mode
      RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2-6, MM.VRes/4-4, 8, Black, Bone     
      TEXT MM.HRes*3/4, MM.VRes*5/8, "RECEIVE", CM, 1, 2, Black, Bone
      RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/2-6, MM.VRes/4-6, 8, Black, Red      
      TEXT MM.HRes*3/4, MM.VRes*7/8, "TRANSMIT", CM, 1, 2, White, Red
      PRINT "Now in TRANSMIT mode..."   ' advise user now in Tx mode
    END IF
  END IF
  SettChg = 1       ' finally set flag to show a change was made
END SUB             ' before returning
  ' -----------------------------------------------------------------
  ' subroutine to listen, in PRX mode (CE pin = 1)
SUB GoListen
  FOR index = 300 TO 1 STEP -1
    FetchnCheckStatus    ' read the nRF24L01 STATUS register, update flags
    IF RxDataRdy <> 0 THEN                ' if Rx data is available
      outData = RRxPLHedr                 ' assemble cmd to read it
      Send32                              ' & go read from Rx FIFO
      RxDataWrd = inData AND &H00FFFFFF   ' extract Rx payload from inData
      ' and then use it to build W_ACK_PAYLOAD  
      outData = ((WAckPLhdr OR RxPipeNum)<< 24) OR RxDataWrd
      Send32                              ' & sent it back to acknowledge
      RestByteOrder                       ' then restore byte order of RxDataWrd
      PRINT "Packet Received = ", HEX$(RRxDataWd, 6) ' and send to console
      ClrStatReg                          ' now clear STATUS reg flag bits
      EXIT SUB                            ' & return
    END IF
  NEXT                                    ' keep looping (until index = 0)
  PRINT "No Packet received - timed out." ' timed out, nothing heard
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to restore byte order of Rx payload (so MS byte is first)
SUB RestByteOrder
  RRxDataWd = 0     ' first clear RRxDataWd for a clean slate
  RRxDataWd = (RxDataWrd AND TSBMask24) >> 16  ' then restore LSB
  RRxDataWd = RRxDataWd OR (RxDataWrd AND NSBMask24) ' then copy NSB
  RRxDataWd = RRxDataWd OR ((RxDataWrd AND LSBMask24) << 16) ' and MSB
END SUB   ' returns with reversed byte order in RRxDataWd
  ' -----------------------------------------------------------------
  ' subroutine to send the current time (ms) in PTX mode (CE pin = 0)
SUB GoSendTime
  PrepmsTime      ' first get current timer value in 24b rev byte order
  outData = (PrepTime OR WTxPLHedr) ' then add Tx header
  Send32               ' send it to Tx FIFO, then
  PULSE 17, 0.015      ' pulse CE line high for 15us to send it off
  FOR index = 300 TO 1 STEP -1
    FetchnCheckStatus       ' fetch STATUS register, update flags
    IF RxDataRdy <> 0 THEN  ' if ACK + Rx data is available
      outData = RRxPLHedr   ' go read it from Rx FIFO
      Send32                ' (Rx data will be returned in inData)
      RxDataWrd = inData AND &H00FFFFFF   ' extract ACK payload from inData
      RestByteOrder        ' return bytes to correct order
      PRINT "Data returned (with ACK) = ", HEX$(RRxDataWd, 6)  ' and print
      ClrStatReg          ' then clear flag bits in STATUS reg
      EXIT SUB            ' and exit from sub
    END IF
    IF MaxRTxbit = 1 THEN ' if MaxRTxbit has been set
      outData = &H2710    ' write to STATUS reg to clear it and thus
      Send16              ' enable further communication
    END IF
  NEXT                    ' otherwise keep looping (until index = 0)
  PRINT "Timed out - No ACK received"    ' print message
  ClrStatReg              ' then clear flag bits in STATUS reg
END SUB                   ' and exit from this sub
  ' ------------------------------------------------------------------
  ' subroutine to get current timer value, mask to 24 bits and assemble
  ' the three bytes in reverse order (LS byte first) for the nRF24L01
SUB PrepmsTime
  RawTVal = TIMER AND &H0000000000FFFFFF    ' strip TIMER val to 24 bits
  PrepTime = (RawTVal AND LSBMask24)<<16    ' mask off LSByte, move up 16
  PTmask = RawTVal AND NSBMask24            ' extract NSByte from RawTVal
  PrepTime = PrepTime OR PTmask             ' & restore in PrepTime
  PTmask = (RawTVal AND TSBMask24)>>16      ' extract TSByte, move down 16
  PrepTime = PrepTime OR PTmask             ' then restore in PrepTime
END SUB   ' return with PrepTime now having bytes in reverse order
  ' -----------------------------------------------------------------
  ' subroutine to reset nRF24L01's STATUS register flag bits, if needed
  ' (by writing a 1 to them only if they are already 1 -- weird!)
SUB ClrStatReg
  FetchnCheckStatus   ' first fetch current status
  outData = &H2700    ' set up cmd header for write to STATUS reg, then
  IF (RxDataRdy = 1) THEN outData = (outData OR &H0040)  ' reset any bits
  IF (TxDatSent = 1) THEN outData = (outData OR &H0020)  ' that need
  IF (MaxRTxbit = 1) THEN outData = (outData OR &H0010)  ' resetting
  Send16                         ' then send back to STATUS reg
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to flush the TX FIFO in PTX mode
SUB FlushTxFIFO
  outData = &HE1    ' set outdata for correct FLUSH_TX command
  Send8             ' then send it
END SUB
  ' -----------------------------------------------------------------
  ' SPI Port function (Revised by Geoff Graham, 4/8/2017)
CFunction SPIPort(integer, integer, integer, integer) integer
00000008
40024800 00442021 40024800 0044102B 1440FFFD 00000000 03E00008 00000000
27BDFFB0 AFBF004C AFBE0048 AFB70044 AFB60040 AFB5003C AFB40038 AFB30034
AFB20030 AFB1002C AFB00028 00808821 00A09021 00C0A021 00E0A821 10800005
8FB30068 10A00004 3C029D00 14C00008 3C109D00 3C029D00 8C420010 00002021
24050002 0040F809 00003021 3C109D00 8E020010 8E240000 24050002 0040F809
00003021 8E020024 8E240000 0040F809 00002821 AFA20018 8E020028 0040F809
8E240000 AFA2001C 8E020024 8E440000 0040F809 24050006 AFA20020 8E020024
8E440000 0040F809 24050005 AFA20024 8E020028 0040F809 8E440000 24170001
0057B804 8FA20064 10400008 3C109D00 8C420000 50400006 8E020024 24030003
5443000D 8E020024 3C109D00 8E020024 8E840000 0040F809 24050005 0040B021
8E020024 8E840000 0040F809 24050006 1000000A 0040F021 8E840000 0040F809
24050006 0040B021 8E020024 8E840000 0040F809 24050005 0040F021 3C029D00
8C420028 0040F809 8E840000 24140001 0054A004 12600002 24110008 8E710000
2631FFFF 32220020 24030001 02238804 02208021 0002800A 0002880B 2402FFFF
2403FFFF AFA20010 12A00005 AFA30014 8EA20000 8EA30004 AFA20010 AFA30014
8FA30060 10600002 0000A821 8C750000 02301025 00009021 10400039 00009821
8FA40010 02241824 8FA60014 02061024 00621025 10400004 8FA30024 8FA20020
10000002 AC570000 AC770000 AED40000 00000000 00000000 00000000 00000000
00000000 00000000 52A00013 00121FC2 02A02021 0411FF6A 00000000 00121FC2
00131040 00122840 8FA60018 8CC40000 8FA6001C 00C42006 30840001 00A49025
00629825 AFD40000 0411FF5D 02A02021 1000000C 00101FC0 00131040 00122840
8FA60018 8CC40000 8FA6001C 00C42006 30840001 00A49025 00629825 AFD40000
00101FC0 00111042 00621025 00101842 00408821 00431025 1440FFC9 00608021
00000000 00000000 00000000 00000000 8FA20064 10400006 00000000 8C420000
10400003 24030002 14430003 02401021 AED40000 02401021 02601821 8FBF004C
8FBE0048 8FB70044 8FB60040 8FB5003C 8FB40038 8FB30034 8FB20030 8FB1002C
8FB00028 03E00008 27BD0050
End CFunction
